<?php

namespace App\Filament\App\Resources;

use App\Filament\App\Resources\GenericCardResource\Pages;
use App\Filament\App\Resources\GenericCardResource\RelationManagers;
use App\Models\GenericCard;
use App\Models\Purchase;
use App\Models\Theme;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Section;
use Illuminate\Support\Facades\Auth;

class GenericCardResource extends Resource
{
    protected static ?string $model = GenericCard::class;

    protected static ?string $navigationIcon = 'heroicon-o-pencil-square';

    protected static ?string $navigationLabel = 'Custom Card';

    protected static ?string $navigationGroup = 'My Folder';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        $userId = Auth::user()->id;

        $purchases = Purchase::where('user_id', $userId)->pluck('theme_code');
        $purchasesAndDefaults = $purchases->push('default_modern');
        $themes = Theme::whereIn('code', $purchasesAndDefaults)->pluck('name', 'code');

        return $form
            ->schema([
                Forms\Components\Select::make('theme_code')
                    ->label(__('Design Theme'))
                    ->native(false)
                    ->options($themes)
                    ->columnSpanFull()
                    ->default('default_modern'),
                Tabs::make('Label')
                    ->tabs([
                        Tabs\Tab::make('Info')
                            ->icon('heroicon-o-envelope-open')
                            ->schema([
                                Forms\Components\TextInput::make('title')
                                    ->autofocus()
                                    ->required()
                                    ->label(__('Title')),
                                Forms\Components\TextInput::make('subtitle')
                                    ->label(__('Subtitle')),
                                Forms\Components\TextInput::make('date')
                                    ->type('date')
                                    ->required()
                                    ->label(__('Date')),
                            ]),
                        Tabs\Tab::make('Details')
                            ->icon('heroicon-o-clock')
                            ->schema([
                                Section::make('Time')
                                    ->schema([
                                        Forms\Components\TextInput::make('start_time')
                                            ->type('time')
                                            ->label(__('Start')),
                                        Forms\Components\TextInput::make('end_time')
                                            ->type('time')
                                            ->label(__('End')),
                                    ])
                                    ->columnSpan(1),
                                Section::make('Location')
                                    ->schema([
                                        Forms\Components\TextInput::make('location_name')
                                            ->required()
                                            ->label(__('Name')),
                                        Forms\Components\TextInput::make('location_coordinate')
                                            ->label(__('Coordinate')),
                                        Forms\Components\Textarea::make('location_address')
                                            ->label(__('Address'))
                                            ->rows(3),
                                    ])
                                    ->columnSpan(1),
                            ])
                            ->columns(2),
                        Tabs\Tab::make('Schedule')
                            ->icon('heroicon-o-calendar-days')
                            ->schema([
                                Forms\Components\Builder::make('schedule')
                                    ->label(__('Schedule'))
                                    ->blocks([
                                        Forms\Components\Builder\Block::make('schedule')
                                            ->schema([
                                                Forms\Components\TextInput::make('time')
                                                    ->label('Time')
                                                    ->type('time')
                                                    ->required(),
                                                Forms\Components\TextInput::make('activity')
                                                    ->label('Activity')
                                                    ->required(),
                                            ])
                                            ->columns(2),
                                    ])
                                    ->addActionLabel('Add Schedule')
                                    ->columnSpanFull(),
                            ]),
                        Tabs\Tab::make('RSVP')
                            ->icon('heroicon-s-user-group')
                            ->schema([
                                Forms\Components\ViewField::make('attendees')
                                    ->label(__('Attendees'))
                                    ->view('filament.forms.components.view-card-attendees')
                                    ->columnSpanFull(),
                                Forms\Components\Builder::make('rsvp_timeslots')
                                    ->label(__('RSVP Timeslots'))
                                    ->blocks([
                                        Forms\Components\Builder\Block::make('timeslots')
                                            ->schema([
                                                Forms\Components\TextInput::make('timeslot')
                                                    ->label('Time')
                                                    ->required(),
                                                Forms\Components\Checkbox::make('on')
                                                    ->label('Active')
                                                    ->default(true)
                                                    ->required(),
                                            ])
                                            ->columns(1),
                                    ])
                                    ->addActionLabel('Add Timeslot')
                                    ->columnSpanFull(),
                            ]),
                        Tabs\Tab::make('Addon')
                            ->icon('heroicon-s-squares-plus')
                            ->schema([
                                Forms\Components\FileUpload::make('bg_music')
                                ->label(__('Background Music'))
                                ->maxSize(8000)
                                ->directory('generic_cards/bg_music')
                                ->disk('public'),
                            ]),
                    ])
                    ->columnSpan('full'),
                Forms\Components\RichEditor::make('body')
                    ->label(__('Body'))
                    ->toolbarButtons([
                        'attachFiles',
                        'bulletList',
                        'h1',
                        'h2',
                        'h3',
                        'italic',
                        'link',
                        'orderedList',
                        'underline',
                    ])
                    ->fileAttachmentsDisk('public')
                    ->fileAttachmentsDirectory('generic_cards/body_images')
                    ->columnSpan('full'),
                Forms\Components\Hidden::make('user_id')
                    ->required()
                    ->default(Auth::user()->id),
            ]);
    }

    public static function table(Table $table): Table
    {
        $userId = Auth::id();
        $table->query(GenericCard::where('user_id', $userId));

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->description(fn(GenericCard $record) => $record->subtitle)
                    ->searchable(),
                Tables\Columns\TextColumn::make('date')
                    ->sortable()
                    ->dateTime('d M Y'),
            ])
            ->filters([
                // Tables\Filters\TrashedFilter::make(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make()
                    ->url(fn(GenericCard $record) => route('cards.show.generic', $record->id)),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                // Tables\Actions\BulkActionGroup::make([
                //     Tables\Actions\DeleteBulkAction::make(),
                //     Tables\Actions\ForceDeleteBulkAction::make(),
                //     Tables\Actions\RestoreBulkAction::make(),
                // ]),
            ])
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListGenericCards::route('/'),
            'create' => Pages\CreateGenericCard::route('/create'),
            'edit' => Pages\EditGenericCard::route('/{record}/edit'),
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->withoutGlobalScopes([
            // SoftDeletingScope::class
        ]);
    }
}
